/*
 * MIT License
 *
 * Copyright (c) 2025 Olzie Development
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.olziedev.olzieplugin.api.scheduler;

import com.olziedev.olzieplugin.api.scheduler.wrapped.chunk.BukkitChunk;
import com.olziedev.olzieplugin.api.scheduler.wrapped.task.FutureTask;
import com.olziedev.olzieplugin.api.scheduler.wrapped.task.PluginTask;
import org.bukkit.Chunk;
import org.bukkit.Location;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Player;
import org.bukkit.event.player.PlayerTeleportEvent;
import org.bukkit.plugin.java.JavaPlugin;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * This class is used to schedule tasks for the plugin.
 *
 * @param <T> The type of plugin task.
 * @param <Z> The type of object to get the plugin task from.
 */
public abstract class PluginScheduler<T, Z> {

    protected JavaPlugin plugin;
    private static ScheduledExecutorService executor;
    private static ScheduledExecutorService singleExecutor;

    /**
     * This constructor is used to create a new plugin scheduler.
     *
     * @param plugin The plugin to use.
     */
    public PluginScheduler(JavaPlugin plugin) {
        this.plugin = plugin;
        setExecutors();
    }

    /**
     * Ths method is used to run a task on the main thread.
     * 
     * @param task The task to run.
     */
    public abstract void runTask(Runnable task);

    /**
     * This method is used to run a task asynchronously.
     * 
     * @param task The task to run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskAsync(Consumer<PluginTask> task);

    /**
     * This method is used to run a task on a single thread.
     * 
     * @param task The task to run.
     * @return The task that was run.
     */
    public PluginTask runSingleTaskAsync(Consumer<PluginTask> task) {
        FutureTask pluginTask = new FutureTask();
        pluginTask.setTask(singleExecutor.submit(() -> task.accept(pluginTask)));
        return pluginTask;
    }

    /**
     * This method is used to run a task on the main thread after a delay.
     * 
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskLater(Consumer<PluginTask> task, long delay);

    /**
     * This method is used to run a task asynchronously after a delay.
     * 
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskLaterAsync(Consumer<PluginTask> task, long delay);

    /**
     * This method is used to run a task on the main thread after a delay and then repeat it.
     *
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @param period The period between each run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskTimer(Consumer<PluginTask> task, long delay, long period);

    /**
     * This method is used to run a task asynchronously after a delay and then repeat it.
     *
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @param period The period between each run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskTimerAsync(Consumer<PluginTask> task, long delay, long period);

    /**
     * This method is used to run a task at a specific location.
     *
     * @param location The location to run the task at.
     * @param task The task to run.
     */
    public void runTaskAtLocation(Location location, Consumer<PluginTask> task) {
        this.runTaskAtLocation(new BukkitChunk(location), task);
    }

    /**
     * This method is used to run a task at a specific location after a delay.
     *
     * @param location The location to run the task at.
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @return The task that was run.
     */
    public PluginTask runTaskAtLocationLater(Location location, Consumer<PluginTask> task, long delay) {
        return this.runTaskAtLocationLater(new BukkitChunk(location), task, delay);
    }

    /**
     * This method is used to run a task at a specific location after a delay and then repeat it.
     *
     * @param location The location to run the task at.
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @param period The period between each run.
     * @return The task that was run.
     */
    public PluginTask runTaskAtLocationTimer(Location location, Consumer<PluginTask> task, long delay, long period) {
        return this.runTaskAtLocationTimer(new BukkitChunk(location), task, delay, period);
    }

    /**
     * This method is used to run a task at a specific chunk.
     *
     * @param chunk The chunk to run the task at.
     * @param task The task to run.
     */
    public abstract void runTaskAtLocation(BukkitChunk chunk, Consumer<PluginTask> task);

    /**
     * This method is used to run a task at a specific chunk after a delay.
     *
     * @param chunk The chunk to run the task at.
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskAtLocationLater(BukkitChunk chunk, Consumer<PluginTask> task, long delay);

    /**
     * This method is used to run a task at a specific chunk after a delay and then repeat it.
     *
     * @param chunk The chunk to run the task at.
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @param period The period between each run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskAtLocationTimer(BukkitChunk chunk, Consumer<PluginTask> task, long delay, long period);

    /**
     * This method is used to run a task at a specific entity.
     *
     * @param entity The entity to run the task at.
     * @param task The task to run.
     */
    public abstract void runTaskAtEntity(Entity entity, Consumer<PluginTask> task);

    /**
     * This method is used to run a task at a specific entity after a delay.
     *
     * @param entity The entity to run the task at.
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskAtEntityLater(Entity entity, Consumer<PluginTask> task, long delay);

    /**
     * This method is used to run a task at a specific entity after a delay and then repeat it.
     *
     * @param entity The entity to run the task at.
     * @param task The task to run.
     * @param delay The delay before the task is run.
     * @param period The period between each run.
     * @return The task that was run.
     */
    public abstract PluginTask runTaskAtEntityTimer(Entity entity, Consumer<PluginTask> task, long delay, long period);

    /**
     * This method is used to teleport a player to a location asynchronously when possible.
     *
     * @param player The player to teleport.
     * @param location The location to teleport the player to.
     * @param task The task to run after the teleport.
     * @param teleportCause The cause of the teleport.
     */
    public abstract void teleportAsync(Player player, Location location, Consumer<Boolean> task, PlayerTeleportEvent.TeleportCause teleportCause);

    /**
     * This method is used to get a chunk asynchronously when possible.
     *
     * @param location The location to get the chunk from.
     * @param task The task to run after the chunk is retrieved.
     */
    public abstract void getChunkAsync(Location location, Consumer<BukkitChunk> task);

    /**
     * This method is used to cancel all the tasks.
     */
    public abstract void cancelAllTasks();

    /**
     * This method is used to obtain a plugin task.
     *
     * @param pluginTaskConsumer The consumer to get the plugin task.
     * @return The plugin task.
     */
    protected abstract T getPluginTask(Consumer<PluginTask> pluginTaskConsumer);

    /**
     * This method is used to obtain a plugin task.
     *
     * @param z The object to get the plugin task from.
     * @return The plugin task.
     */
    protected abstract PluginTask getPluginTask(Z z);

    /**
     * This method is used to schedule a runnable to run after a delay.
     *
     * @param runnable The runnable to run.
     * @param date The date to run the runnable.
     */
    public static ScheduledFuture<?> schedule(Runnable runnable, Date date) {
        long delay = Math.max(50, date.getTime() - System.currentTimeMillis());
        return executor.schedule(() -> {
            try {
                runnable.run();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }, delay, TimeUnit.MILLISECONDS);
    }

    /**
     * This method is used to setup the executors.
     */
    public static void setExecutors() {
        if (executor != null) executor.shutdownNow();
        if (singleExecutor != null) singleExecutor.shutdownNow();
        
        executor = Executors.newScheduledThreadPool(15);
        singleExecutor = Executors.newSingleThreadScheduledExecutor();
    }
}
