/*
 * MIT License
 *
 * Copyright (c) 2024 Olzie Development
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.olziedev.olzieplugin.api.expansion;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.olziedev.olzieplugin.api.OlziePluginAPI;
import org.bukkit.configuration.file.FileConfiguration;

import java.io.File;

/**
 * Represents an expansion for it to be loaded and used in {@link ExpansionRegistry}.
 */
public abstract class PluginExpansion {

    /**
     * This variable is initialised when the expansion has been loaded and returns the PlayerWarps plugin instance.
     */
    @JsonIgnore
    public OlziePluginAPI plugin;
    /**
     * This variable is initialised when the expansion has been loaded and returns the registry it was loaded in.
     */
    @JsonIgnore
    public ExpansionRegistry registry;
    /**
     * This variable is initialised when the expansion has been loaded and returns the PlayerWarps expansion config instance.
     */
    @JsonIgnore
    public FileConfiguration expansionConfig;
    /**
     * This variable is initialised when the expansion has been loaded and returns the PlayerWarps config instance.
     */
    @JsonIgnore
    public FileConfiguration config;
    /**
     * This variable is initialised when the expansion has been loaded and returns if the expansion is integrated into the plugin
     * or if it has been manually registered using {@link ExpansionRegistry#registerExpansion(PluginExpansion)}
     */
    public boolean integrated;
    /**
     * This variable is initialised when the expansion has been loaded and returns the version of the expansion.
     */
    public String version;

    /**
     * This variable is initialised when the expansion has been loaded and returns the file of the expansion.
     * This can return null if the expansion is integrated into the plugin.
     */
    @JsonIgnore
    public File file;

    /**
     * @return If the expansion is enabled in the configuration file and is loaded onto the server.
     */
    public abstract boolean isEnabled();

    /**
     * @return The name of the expansion that will be used in areas of the plugin.
     */
    public abstract String getName();

    /**
     * This method will be initialised if {@link #isEnabled()} returns true on plugin enable.
     */
    public abstract void onLoad();

    /**
     * The method that will be initialised before the entire plugin is loaded.
     */
    public void onPreLoad() {}

    /**
     *  This method to unload the expansion from the registry.
     */
    public void unload() {
        this.registry.unregisterExpansion(this.getName());
    }

    /**
     *  This method will be initialised when the addon is unloading.
     */
    public void onUnload() {}
}
