/*
 * MIT License
 *
 * Copyright (c) 2024 Olzie Development
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.olziedev.olzieplugin.api.scheduler.wrapped.chunk;

import org.bukkit.Chunk;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.block.Block;

/**
 * Represents a chunk in the world.
 */
public class BukkitChunk {

    private Chunk chunk;

    private final World world;
    private final int x;
    private final int z;
    /**
     * @param chunk The chunk to wrap.
     */
    public BukkitChunk(Chunk chunk) {
        this.chunk = chunk;
        this.world = chunk.getWorld();
        this.x = chunk.getX();
        this.z = chunk.getZ();
    }

    public BukkitChunk(Location location) {
        this(location.getWorld(), location.getBlockX() >> 4, location.getBlockZ() >> 4);
    }

    /**
     * @param x The x coordinate of the chunk.
     * @param z The z coordinate of the chunk.
     */
    public BukkitChunk(World world, int x, int z) {
        this.world = world;
        this.x = x;
        this.z = z;
    }

    /**
     * Gets the block at the specified coordinates.
     *
     * @param x The x coordinate of the block.
     * @param y The y coordinate of the block.
     * @param z The z coordinate of the block.
     * @return The block at the specified coordinates.
     */
    public Block getBlock(int x, int y, int z) {
        if (chunk == null) {
            return null;
        }
        return chunk.getBlock(x & 15, y, z & 15);
    }

    /**
     * @return The world that the chunk is in.
     */
    public World getWorld() {
        return this.world;
    }

    /**
     * @return The x coordinate of the chunk.
     */
    public int getX() {
        return this.x;
    }

    /**
     * @return The z coordinate of the chunk.
     */
    public int getZ() {
        return this.z;
    }
}
