/*
 * MIT License
 *
 * Copyright (c) 2024 Olzie Development
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.olziedev.olzieplugin.api.expansion;

import net.milkbowl.vault.chat.Chat;
import net.milkbowl.vault.economy.Economy;
import net.milkbowl.vault.permission.Permission;
import org.bukkit.OfflinePlayer;
import org.bukkit.configuration.ConfigurationSection;

import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

/**
 * Handles and manages all expansions for the plugin.
 */
public abstract class ExpansionRegistry {

    /**
     * Register an pluginExpansion to the registry.
     *
     * @param pluginExpansion The pluginExpansion instance.
     */
    public abstract void registerExpansion(PluginExpansion pluginExpansion);

    /**
     * Initialize the placeholders for the pluginExpansion.
     *
     * @param pluginExpansion The pluginExpansion instance.
     */
    public abstract void innitPlaceholders(PluginExpansion pluginExpansion);

    /**
     * Execute a consumer action on the specified implementation.
     *
     * @param clazz  The expansion implementation to execute the action for.
     * @param action Action that will be executed for each expansion.
     * @param <T>    The expansion implementation.
     */
    public abstract <T extends PluginExpansion> void executeExpansionAction(Class<T> clazz, Consumer<? super T> action);

    /**
     * Unregister an expansion from the registry.
     *
     * @param name The expansion name from {@link PluginExpansion#getName()}.
     */
    public abstract void unregisterExpansion(String name);

    /**
     * Unregisters all the loaded expansions from the registry.
     */
    public abstract void shutdownExpansions();

    /**
     * Reload all the expansions from the registry.
     */
    public abstract void reloadExpansions();

    /**
     * Retrieve a list of all the expansions based on the implementation specified.
     *
     * @param clazz The expansion implementation to return.
     * @param <T>   The expansion implementation.
     * @return List of all the loaded expansions based on the implementation.
     */
    public abstract <T extends PluginExpansion> List<T> getExpansions(Class<T> clazz);

    /**
     * Retrieve an expansion based on the class specified.
     *
     * @param clazz The expansion to return.
     * @param <T>  The expansion class.
     * @return The expansion instance.
     */
    public abstract <T extends PluginExpansion> T getExpansion(Class<T> clazz);

    /**
     * Retrieve an expansion based on the name specified.
     *
     * @param name The name of the expansion to retrieve.
     * @return The expansion instance.
     */
    public abstract PluginExpansion getExpansion(String name);

    /**
     * Retrieve a list of all the expansions.
     *
     * @return List of all the loaded expansions in the registry.
     */
    public List<PluginExpansion> getExpansions() {
        return this.getExpansions(PluginExpansion.class);
    }

    /**
     * Retrieve a stream of all the expansions based on the implementation specified.
     *
     * @param clazz The expansion implementation to return.
     * @param <T>  The expansion implementation.
     * @return Stream of all the loaded expansions based on the implementation.
     */
    public abstract  <T extends PluginExpansion> Stream<T> getExpansionStream(Class<T> clazz);


    /**
     * Retrieve the integrated permission provider in this registry from Vault.
     *
     * @return Permission provider from the internal Vault expansion.
     */
    public abstract Permission getVaultExpansionPermission();

    /**
     * Retrieve the integrated permission provider in this registry from Vault.
     *
     * @return Economy provider from the internal Vault expansion.
     */
    public abstract Economy getVaultExpansionEconomy();

    /**
     * Retrieve the integrated permission provider in this registry from Vault.
     *
     * @return Chat provider from the internal Vault expansion.
     */
    public abstract Chat getVaultExpansionChat();

    public abstract boolean noPlaceholderAPICondition(ConfigurationSection section, Function<String, String> function, OfflinePlayer player);;

}
