/*
 * MIT License
 *
 * Copyright (c) 2025 Olzie Development
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.olziedev.olzieplugin.api.internal;

import com.olziedev.olzieplugin.api.OlziePluginAPI;
import com.olziedev.olzieplugin.api.utils.LogLevel;
import io.papermc.paper.event.player.AsyncChatEvent;
import net.kyori.adventure.text.Component;
import net.kyori.adventure.text.TextComponent;
import net.kyori.adventure.text.format.TextDecoration;
import net.kyori.adventure.text.minimessage.MiniMessage;
import net.kyori.adventure.text.serializer.legacy.LegacyComponentSerializer;
import org.bukkit.ChatColor;
import org.bukkit.inventory.meta.ItemMeta;

import java.util.List;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * This class is only to be supported with Paper and any of the adventure imports SHOULD NOT BE RELOCATED!
 */
public class PaperUtils {

    public static void setLore(List<String> lore, ItemMeta itemMeta, Predicate<String> filter, OlziePluginAPI olziePluginAPI) {
        Pattern pattern = Pattern.compile("([a-zA-Z0-9_§&]+(\\.[a-zA-Z0-9_§&]+)+)");
        List<Component> list = lore.stream().filter(filter).map(x -> {
            Component component;
            try {
                component = MiniMessage.miniMessage().deserialize(x);
            } catch (Throwable e) {
                component = LegacyComponentSerializer.builder().build().deserialize(x);
            }
            component = component.decorationIfAbsent(TextDecoration.ITALIC, TextDecoration.State.FALSE);
            Matcher matcher = pattern.matcher(x);
            olziePluginAPI.debug("Matching: " + x, LogLevel.MAJOR);
            while (matcher.find()) {
                olziePluginAPI.debug("Matched Before: " + matcher.group(0), LogLevel.MAJOR);
                component = component.replaceText((x2 -> x2.matchLiteral(ChatColor.stripColor(matcher.group(0))).replacement((x4, x3) -> {
                    olziePluginAPI.debug("Replacing text: " + x3.content() + " with " + matcher.group(0));
                    return Component.translatable(x3.content(), Component.text(matcher.group(0)));
                })));
                olziePluginAPI.debug("Matched After: " + matcher.group(0), LogLevel.MAJOR);
            }
            return component;
        }).collect(Collectors.toList());
        if (!list.isEmpty()) itemMeta.lore(list);
    }

    public static void setDisplayName(String s, ItemMeta itemMeta) {
        Component component;
        try {
            component = MiniMessage.miniMessage().deserialize(s);
        } catch (Throwable e) {
            component = LegacyComponentSerializer.builder().build().deserialize(s);
        }
        try {
            itemMeta.displayName(component.decorationIfAbsent(TextDecoration.ITALIC, TextDecoration.State.FALSE));
        } catch (NoSuchMethodError ex) {
            itemMeta.customName(component.decorationIfAbsent(TextDecoration.ITALIC, TextDecoration.State.FALSE));
        }
    }

    public static String getMessage(AsyncChatEvent event) {
        Component component = event.message();
        if (!(component instanceof TextComponent)) return "";

        return ((TextComponent) component).content();
    }
}
